#![feature(iter_intersperse)]
extern crate clap;

#[macro_use]
extern crate log;

mod exec;
mod error;
mod mlir;
mod tools;
use std::{collections::VecDeque, env::current_exe, ffi::OsStr, fs::File, io::{Read, Write}, path::{Path, PathBuf}};

use clap::*;
use error::*;
use isq_version::ISQVersion;
use itertools::Itertools;
use tools::{isq_opt, isq_simulator::{self, Backend, ISQSimulator, ISQSimulatorConfig}, isqc1, llvm::{llc, lld, llvm_link, mlir_translate, opt::{self, OptFlags}}, ISQBinTool};

//use crate::tools::isqc1::resolve::resolve_isqc1_output;
//use crate::tools::mlir::resolve_mlir_output;

#[derive(Parser)]
#[clap(name = "isQ Compiler", version = ISQVersion::build_semver(),
long_version = Arguments::long_version(), 
about = "isQ Compiler")]
pub struct Arguments {
    #[clap(subcommand)]
    command: Commands,
}
impl Arguments{
    fn long_version()->String{
        format!("{}\nGit revision: {}", ISQVersion::build_semver(),
        ISQVersion::build_rev())
    }
}

#[derive(Copy, Clone, PartialEq, Eq, PartialOrd, Ord, ValueEnum)]
pub enum EmitMode {
    MLIR,
    MLIRQIR,
    LLVM,
    MLIROptimized,
    Binary,
    Out
}
#[derive(Copy, Clone, PartialEq, Eq, PartialOrd, Ord, ValueEnum)]
pub enum CompileTarget{
    QIR,
    OpenQASM3,
    QCIS,
    UNI,
    EQASM
}

#[derive(Subcommand)]
pub enum Commands{
    Compile{
        input: String,
        #[clap(long, short)]
        output: Option<String>,
        #[arg(long, short='O', value_parser = value_parser!(u64).range(0..=3))]
        opt_level: Option<u64>,
        #[clap(long, value_enum, default_value = "out")]
        emit: EmitMode,
        #[clap(long, value_enum, default_value = "qir")]
        target: CompileTarget,
        #[clap(long)]
        qcis_config: Option<String>,
        #[clap(long, short='I', action=ArgAction::Append)]
        inc_path: Option<Vec<String>>,
        #[clap(long, short, action=ArgAction::Append, allow_negative_numbers(true))]
        int_par: Option<Vec<i64>>,
        #[clap(long, short, action=ArgAction::Append, allow_negative_numbers(true))]
        double_par: Option<Vec<f64>>
        //#[clap(long, default_value = "native")]
        //llvmTriple: String
    },
    #[clap(group(
        ArgGroup::new("simulator_type")
            .required(false)
            .args(&["cuda", "qcis"]),
    ))]
    Simulate{
        #[clap(required(true))]
        qir_object: String,
        #[clap(long)]
        cuda: Option<usize>,
        #[clap(long)]
        qcis: bool,
        #[clap(long)]
        shots: Option<i64>,
        #[clap(long)]
        debug: bool,
        #[clap(long, short, action=ArgAction::Append, allow_negative_numbers(true))]
        int_par: Option<Vec<i64>>,
        #[clap(long, short, action=ArgAction::Append, allow_negative_numbers(true))]
        double_par: Option<Vec<f64>>,
        #[clap(long, short, default_value = "1")]
        np: i64,
        #[clap(long, default_value = "25")]
        qn: usize,
        #[clap(long)]
        probs: bool
    },
    Exec{
        #[clap(action=ArgAction::Append, required(true))]
        exec_command: Vec<String>
    },
    Run{
        input: String,
        #[clap(long)]
        shots: Option<i64>,
        #[clap(long)]
        debug: bool,
        #[clap(long, short, default_value = "1")]
        np: i64,
        #[clap(long, default_value = "25")]
        qn: usize,
        #[clap(long)]
        probs: bool
    }
}


fn resolve_input_path<'a>(input: &'a str, extension: &str)->miette::Result<(&'a Path, PathBuf)>{
    let input_path = Path::new(input);
            
    if input_path.extension()!=Some(OsStr::new("isq")){
        return Err(BadExtensionError)?;
    }
    
    let output = input_path.with_extension(extension);
    return Ok((input_path, output));
}

fn main()->miette::Result<()> {
    env_logger::init();
    info!("isQ Compiler {}", ISQVersion::build_semver());
    let cli = Arguments::parse();
    #[cfg(target_os = "windows")]
    {
        // get root from ${root}/bin/${current_exe}.
        if std::env::var("ISQ_ROOT").is_err(){
            let mut current_exe = std::env::current_exe().map_err(io_error_when("Guessing ISQ_ROOT"))?;
            current_exe.pop();
            current_exe.pop();
            std::env::set_var("ISQ_ROOT", current_exe);
        }
    }
    let root = std::env::var("ISQ_ROOT").map_err(|_| NoISQv2RootError)?;
    let mut queue = VecDeque::from([cli.command]);
    while !queue.is_empty() {
        let cmd = queue.pop_front().unwrap();
        match cmd{
            Commands::Run{input, shots, debug, np, qn, probs}=>{
                let (input_path, default_output_path) = resolve_input_path(&input, "so")?;
                let so_path = default_output_path.to_string_lossy().to_string();
                queue.push_back(Commands::Compile { 
                    input: input_path.to_string_lossy().to_string(), output: Some(so_path.clone()),
                    opt_level: None, emit: EmitMode::Out, target: CompileTarget::QIR, qcis_config: None,
                    inc_path: None, int_par: None, double_par: None 
                });
                queue.push_back(Commands::Simulate { 
                    qir_object: so_path, cuda: None, qcis: false, shots: shots, debug: debug, int_par: None, 
                    double_par: None, np: np, qn: qn, probs: probs
                })
            }
            Commands::Compile{input, output, opt_level, emit, target, qcis_config, inc_path, int_par, double_par}=>'command:{
                
                let simulator_config = ISQSimulatorConfig::new();
                let (input_path, default_output_path) = resolve_input_path(&input, &match emit{
                    EmitMode::Binary=>simulator_config.dllextension()?,
                    EmitMode::Out=> simulator_config.dllextension()?,
                    EmitMode::LLVM=>"ll".to_owned(),
                    EmitMode::MLIR=>"mlir".to_owned(),
                    EmitMode::MLIRQIR=>"ll.mlir".to_owned(),
                    EmitMode::MLIROptimized=>"opt.mlir".to_owned()
                })?;
                if ISQBinTool::isq_tool_debug(){
                    println!("input={:?} output={:?}", input_path, default_output_path);
                }
                let output = output.map_or_else(||{
                    default_output_path
                }, |x| {PathBuf::from(x)});

                let f_in_path = input_path.canonicalize().unwrap();
                let input_file = crate::tools::File::ExternalFile(f_in_path.to_str().ok_or(MalformedUTF8Error)?.to_string());
                let isqc1 = isqc1::ISQC1::new(&inc_path.into_iter().flatten().collect_vec());
                let mlir = isqc1.compile_with_flags(&input_file, target == CompileTarget::QCIS || target == CompileTarget::UNI)?;
                
                if let EmitMode::MLIR = emit{
                    mlir.copy_to(output)?;
                    break 'command;
                }
                let isq_opt = isq_opt::ISQOpt::new();
                let flags = match target {
                    CompileTarget::QCIS => isq_opt::ISQOpt::qcis_flags(false),
                    CompileTarget::UNI => isq_opt::ISQOpt::qcis_flags(true),
                    CompileTarget::OpenQASM3 => isq_opt::ISQOpt::qasm_flags(),
                    CompileTarget::EQASM => isq_opt::ISQOpt::qasm_flags(),
                    _ => isq_opt::ISQOpt::simulator_flags(),
                };
                let optimized_mlir = isq_opt.optimize(&[flags], &mlir)?;
                if let EmitMode::MLIROptimized = emit{
                    optimized_mlir.copy_to(output)?;
                    break 'command;
                }

                if let CompileTarget::EQASM = target{
                    let eqasm_mlir = isq_opt.codegen_eqasm(&optimized_mlir)?;
                    eqasm_mlir.copy_to(output)?;
                    break 'command;
                }

                if let CompileTarget::OpenQASM3 = target{
                    let qasm3_mlir = isq_opt.codegen_qasm3(&optimized_mlir)?;
                    qasm3_mlir.copy_to(output)?;
                    break 'command;
                }

                let llvm_mlir = isq_opt.lower_to_llvm_mlir(&optimized_mlir)?;

                if let EmitMode::MLIRQIR = emit{
                    llvm_mlir.copy_to(output)?;
                    break 'command;
                }
                let mlir_translate = mlir_translate::MLIRTranslate::new();

                let llvm = mlir_translate.mlir_to_llvmir(&llvm_mlir)?;

                if let EmitMode::LLVM = emit{
                    llvm.copy_to(output)?;
                    break 'command;
                }
                let header = simulator_config.llvm_ir_header()?;
                let llvm_link = llvm_link::LLVMLink::new();
                
                let llvm_link_args = simulator_config.llvm_link_flags()?;
                let simulator_stub = tools::File::ExternalFile(simulator_config.llvm_link_stub()?);
                let llvm_bc = llvm_link.link(&[
                    &llvm, &simulator_stub], &llvm_link_args, Some(&header))?;
                let llvm_opt = opt::Opt::new();
                let mut opt_args: Vec<OptFlags> = Vec::new();
                match opt_level{
                    None=>{}
                    Some(0)=>{}
                    Some(1)=>{opt_args.push(OptFlags::O1);}
                    Some(2)=>{opt_args.push(OptFlags::O2);}
                    Some(3)=>{opt_args.push(OptFlags::O3);}
                    _=>{

                    }
                }
                let optimized_llvm_bv = llvm_opt.run(&llvm_bc, &opt_args)?;
                let llc = llc::LLC::new();
                let object_file = llc.run(&optimized_llvm_bv)?;
                let lld = lld::LLD::new();
                
                let binary_file = lld.run(&object_file, &simulator_config.ldflags()?)?;

                
                binary_file.copy_to(output)?;
                // post-processing.
                if let EmitMode::Out = emit{
                    if CompileTarget::QCIS == target || CompileTarget::UNI == target {
                        let mut simulator = isq_simulator::ISQSimulator::new();
                        
                        // get parameters
                        let par_int = match int_par{
                            Some(x) => x,
                            None => vec![]
                        };
                        let par_double = match double_par{
                            Some(x) => x,
                            None => vec![]
                        };
                        simulator.parameters.param_int = par_int;
                        simulator.parameters.param_double = par_double;
                        let qcis_route_config = qcis_config.as_ref().map(|x| crate::tools::File::ExternalFile(x.clone()));
                        let qcis_output_file = simulator.generate_qcis(&binary_file, qcis_route_config.as_ref())?;
                        // run simulator 
                        let (_, qcis_output_path) = resolve_input_path(&input, "qcis")?;
                        qcis_output_file.copy_to(&qcis_output_path)?;
                        if CompileTarget::UNI == target {
                            let f = crate::tools::File::ExternalFile(qcis_output_path.to_str().ok_or(MalformedUTF8Error)?.to_owned());
                            let mut fh = f.open()?;
                            let mut buf = String::new();
                            fh.read_to_string(&mut buf).map_err(io_error_when("Reading QCIS Universal"))?;
                            println!("{}", buf);
                        }
                    }else if let CompileTarget::OpenQASM3 = target{
                        todo!("openqasm post-generation");
                    }
                }

            }
            Commands::Simulate{qir_object, cuda, qcis, shots, debug, int_par, double_par, np, qn,probs}=>{

                let qir_object = if qir_object.starts_with("/"){
                    qir_object
                }else{
                    format!("./{}", qir_object)
                };
                let qir_file = crate::tools::File::ExternalFile(qir_object);
                let mut simulator = ISQSimulator::new();
                if let Some(x)=cuda{
                    simulator.backend = Backend::CUDA { max_qubits: x };
                }else if qcis{
                    simulator.backend = Backend::QCIS;
                }
                if let Some(x)=shots{
                    simulator.shots = x;
                }
                if debug{
                    simulator.debug = true;
                }
                if probs{
                    simulator.probs = true;
                }

                // get parameters
                simulator.parameters.param_int = match int_par{
                    Some(x) => x,
                    None => vec![]
                };
                simulator.parameters.param_double = match double_par{
                    Some(x) => x,
                    None => vec![]
                };
                simulator.np = np;
                simulator.qn = qn;
                simulator.run(&qir_file)?;
            }
            Commands::Exec{exec_command}=>{
                exec::raw_exec_command(&root, &exec_command[0], &exec_command[1..], ).map_err(IoError)?;
            }
        }
    }
    return Ok(())
}
