use crate::io_error_when;
use self::resolve::resolve_mlir_output;

use super::{ext::{eqasm, mlir_src, qasm3}, File, ISQBinTool};
use std::io::Write;
pub mod resolve;
pub struct ISQOpt{
    tool: ISQBinTool
}
#[derive(Clone)]
pub struct ISQOptPipeline(Vec<ISQOptPassList>);
impl ISQOptPipeline{
    fn to_flag(&self)->String{
        let mut s = String::new();
        s.push_str("-pass-pipeline=");
        s.extend(self.0.iter().map(|p| p.to_flag()).intersperse(",".to_owned()));
        s
    }
}
#[derive(Clone)]
pub struct ISQOptPassList{
    operation: String,
    passes: Vec<ISQOptPass>
}
impl ISQOptPassList{
    fn to_flag(&self)->String{
        let mut s = String::new();
        s.push_str(&self.operation);
        s.push_str("(");
        s.extend(self.passes.iter().map(|p| p.to_flag()).intersperse(",".to_owned()));
        s.push_str(")");
        s
    }
}
#[derive(Clone)]
pub enum ISQOptPass{
    Single(ISQOptPassLeaf),
    Nested(ISQOptPassList)
}
impl ISQOptPass{
    pub fn to_flag(&self)->String{
        match self{
            ISQOptPass::Single(p) => p.to_flag(),
            ISQOptPass::Nested(p) => p.to_flag()
        }
    }
}
#[derive(Clone)]
pub struct ISQOptPassLeaf{
    name: String
}
impl ISQOptPassLeaf{
    fn to_flag(&self)->String{
        self.name.to_owned()
    }
}
pub mod passes{
    use super::*;
    fn underscore_to_hyphens(s: &str)->String{
        s.replace("_", "-")
    }
    macro_rules! simple_pass {
        ($x: ident) => {
            pub fn $x()->ISQOptPassLeaf{
                ISQOptPassLeaf{
                    name: underscore_to_hyphens(stringify!($x))
                }
            }
        };
    }
    macro_rules! simple_pass_list {
        ($($x: ident),*) => {
            $(
                simple_pass!($x);
            )*
        }
    }
    simple_pass_list!(instantiate_template, cse, isq_state_preparation,isq_remove_reset,isq_oracle_decompose,isq_permutation_decompose,isq_lower_switch,isq_canonicalize,isq_gen_deriving_inverse,isq_recognize_famous_gates,isq_eliminate_neg_ctrl,isq_pure_gate_detection,isq_fold_decorated_gates,isq_inline,canonicalize,isq_decompose_ctrl_u3,isq_convert_famous_rot,isq_decompose_known_gates_qsd,isq_remove_trivial_sq_gates,isq_target_qcis,isq_expand_decomposition,isq_cancel_redundant, isq_to_qcis_set, convert_math_to_llvm,
    isq_remove_gphase, lower_affine,
    isq_lower_to_qir_rep, arith_expand,
    expand_strided_metadata, memref_expand,
    convert_math_to_funcs, isq_lower_qir_rep_to_llvm,
    symbol_dce, llvm_legalize_for_export, global_thread_local
    );
}

impl ISQOpt{
    pub fn new()->Self{
        Self{
            tool: ISQBinTool::new("isq-opt")
        }
    }
    pub fn qcis_flags(use_universal_gateset: bool)->ISQOptPipeline{
        let mut passes = vec![
            passes::instantiate_template(), passes::cse(), passes::isq_state_preparation(), passes::isq_remove_reset(), passes::isq_oracle_decompose(), passes::isq_permutation_decompose(), passes::isq_lower_switch(), passes::isq_canonicalize(), passes::canonicalize(), passes::isq_gen_deriving_inverse(), passes::isq_recognize_famous_gates(), passes::isq_eliminate_neg_ctrl(), passes::canonicalize(), passes::cse(), passes::isq_pure_gate_detection(), passes::canonicalize(), passes::isq_fold_decorated_gates(), passes::isq_inline(), passes::canonicalize(), passes::isq_decompose_ctrl_u3(), passes::isq_convert_famous_rot(), passes::isq_decompose_known_gates_qsd(), passes::isq_remove_trivial_sq_gates()
        ];
        if !use_universal_gateset{
            passes.push(passes::isq_to_qcis_set());
        }
        passes.extend([passes::isq_target_qcis(),passes::isq_expand_decomposition(),passes::canonicalize(),passes::cse(),passes::canonicalize(),passes::cse()]);
        ISQOptPipeline(vec![ISQOptPassList{
            operation: "builtin.module".to_owned(),
            passes: passes.into_iter().map(ISQOptPass::Single).collect()
        }])
    }
    pub fn simulator_flags()->ISQOptPipeline{
        let passes = vec![
            passes::instantiate_template(), passes::canonicalize(), passes::cse(), passes::isq_state_preparation(), passes::isq_oracle_decompose(), passes::isq_permutation_decompose(), passes::isq_lower_switch(), passes::canonicalize(), passes::isq_gen_deriving_inverse(), passes::isq_recognize_famous_gates(), passes::isq_eliminate_neg_ctrl(), passes::isq_convert_famous_rot(), passes::canonicalize(), passes::cse(), passes::isq_pure_gate_detection(), passes::canonicalize(), passes::isq_fold_decorated_gates(), passes::isq_inline(), passes::canonicalize(), passes::isq_decompose_ctrl_u3(), passes::isq_convert_famous_rot(), passes::isq_decompose_known_gates_qsd(), passes::isq_remove_trivial_sq_gates(), passes::isq_expand_decomposition(), passes::canonicalize(), passes::cse()
        ];
        ISQOptPipeline(vec![ISQOptPassList{
            operation: "builtin.module".to_owned(),
            passes: passes.into_iter().map(ISQOptPass::Single).collect()
        }])
    }
    pub fn lower_flags()->ISQOptPipeline{
        let mut passes = [
            passes::cse(), passes::isq_remove_gphase(), passes::lower_affine(), passes::isq_lower_to_qir_rep(), passes::cse(), passes::canonicalize()].map(ISQOptPass::Single).into_iter().collect::<Vec<_>>();
        passes.push(ISQOptPass::Nested(ISQOptPassList{
            operation: "func.func".to_owned(),
            passes: vec![ISQOptPass::Single(passes::convert_math_to_llvm())]
        }));
        passes.extend([passes::arith_expand(), passes::expand_strided_metadata(), passes::memref_expand(), passes::convert_math_to_funcs(), passes::isq_lower_qir_rep_to_llvm(), passes::canonicalize(), passes::cse(), passes::symbol_dce(), passes::llvm_legalize_for_export(), passes::global_thread_local()].map(ISQOptPass::Single).into_iter().collect::<Vec<_>>());
        ISQOptPipeline(vec![ISQOptPassList{
            operation: "builtin.module".to_owned(),
            passes
        }])
    }
    pub fn qasm_flags()->ISQOptPipeline{
        let passes = vec![
            passes::instantiate_template(), passes::cse(), passes::isq_state_preparation(), passes::isq_oracle_decompose(), passes::isq_permutation_decompose(), passes::isq_lower_switch(), passes::canonicalize(), passes::isq_gen_deriving_inverse(), passes::isq_recognize_famous_gates(), passes::isq_eliminate_neg_ctrl(), passes::canonicalize(), passes::cse(), passes::isq_pure_gate_detection(), passes::canonicalize(), passes::isq_fold_decorated_gates(), passes::isq_inline(), passes::canonicalize(), passes::isq_decompose_ctrl_u3(), passes::isq_decompose_known_gates_qsd(), passes::isq_remove_trivial_sq_gates(), passes::isq_expand_decomposition(), passes::isq_cancel_redundant(), passes::canonicalize(), passes::cse()
        ];
        ISQOptPipeline(vec![ISQOptPassList{
            operation: "builtin.module".to_owned(),
            passes: passes.into_iter().map(ISQOptPass::Single).collect()
        }])
    }
    pub fn optimize(&self, opt_flags: &[ISQOptPipeline], source: &File)->miette::Result<File>{
        let optimized_mlir_output = self.tool.tempfile(mlir_src())?;
        let mut flags = vec![];
        flags.extend(opt_flags.iter().map(|x| x.to_flag()));
        flags.extend(["--mlir-print-debuginfo", "--format-out"].map(|x| x.to_owned()));
        flags.push(source.path()?.to_owned());
        let isq_opt_out = self.tool.call(&flags)?;
        let resolved_mlir_opt = resolve::resolve_mlir_output(&isq_opt_out, "Internal compiler error: isq-opt failed.".to_owned())?;
        let mut f = optimized_mlir_output.open()?;
        write!(f, "{}", resolved_mlir_opt).map_err(io_error_when("Writing MLIR"))?;
        Ok(optimized_mlir_output)
    }
    pub fn codegen_eqasm(&self, source: &File)->miette::Result<File>{
        let eqasm_output = self.tool.tempfile(eqasm())?;
        let mut flags = vec![];
        flags.extend(["--target=eqasm", "--format-out"].map(str::to_owned));
        flags.push(source.path()?.to_owned());
        let isq_opt_out = self.tool.call(&flags)?;
        let qasm3 = resolve_mlir_output(&isq_opt_out, "Internal compiler error: isq-opt eQASM codegen failed.".to_owned())?;
        let mut f = eqasm_output.open()?;
        write!(f, "{}", qasm3).map_err(io_error_when("Writing eQASM"))?;
        Ok(eqasm_output)
    }
    pub fn codegen_qasm3(&self, source: &File)->miette::Result<File>{
        let qasm3_output = self.tool.tempfile(qasm3())?;
        let mut flags = vec![];
        flags.extend(["--target=openqasm3", "--format-out"].map(str::to_owned));
        flags.push(source.path()?.to_owned());
        let isq_opt_out = self.tool.call(&flags)?;
        let qasm3 = resolve_mlir_output(&isq_opt_out, "Internal compiler error: isq-opt OpenQASM3 codegen failed.".to_owned())?;
        let mut f = qasm3_output.open()?;
        write!(f, "{}", qasm3).map_err(io_error_when("Writing OpenQASM3"))?;
        Ok(qasm3_output)
    }
    pub fn lower_to_llvm_mlir(&self, source: &File)->miette::Result<File>{
        self.optimize(&[ISQOpt::lower_flags()], source)
    }
}