use crate::io_error_when;
use std::io::Write;
use super::{ext::qcis, File, ISQBinTool};
pub struct ISQSimulatorConfig(ISQBinTool);

impl ISQSimulatorConfig {
    pub fn new()->Self{
        Self(ISQBinTool::new("isq-simulator-config"))
    }
    pub fn get(&self,arg: &str)->miette::Result<String>{
        let result = self.0.call(&[arg])?.trim().to_string();
        if ISQBinTool::isq_tool_debug(){
            println!("ISQSimulatorConfig fetching {} returning {}", arg, result);
        }
        Ok(result)
    }
    pub fn llvm_ir_header(&self)->miette::Result<String>{
        Ok(format!("{}\n\n", self.get("llvmir-header")?))
    }
    pub fn target_triple(&self)->miette::Result<String>{
        self.get("target-triple")
    }
    pub fn llvm_link_flags(&self)->miette::Result<Vec<String>>{
        Ok(self.get("llvm-link-flags")?.split_whitespace().map(|x| x.to_owned()).collect())
    }
    pub fn ldflags(&self)->miette::Result<Vec<String>>{
        Ok(self.get("ld-flags")?.split_whitespace().map(|x| x.to_owned()).collect())
    }
    pub fn dllsuffix(&self)->miette::Result<String>{
        self.get("dll-suffix")
    }
    pub fn dllextension(&self)->miette::Result<String>{
        let suffix = self.dllsuffix()?;
        // remove prefixing dot
        Ok(suffix.trim_start_matches('.').to_string())
    }
    pub fn data_layout(&self)->miette::Result<String>{
        self.get("data-layout")
    }
    pub fn llvm_link_stub(&self)->miette::Result<String>{
        self.get("llvm-link-stub")
    }


}

pub struct Parameters{
    pub param_int: Vec<i64>,
    pub param_double: Vec<f64>
}
impl Default for Parameters{
    fn default() -> Self {
        Self{
            param_int: vec![],
            param_double: vec![]
        }
    }
}
pub enum Backend{
    CUDA{
        max_qubits: usize
    },
    Naive,
    QCIS,
    QCISGen
}
impl Backend{
    pub fn to_args(&self)->Vec<String>{
        match self{
            Backend::CUDA { max_qubits } => {
                vec!["--cuda".to_owned(), max_qubits.to_string()]
            },
            Backend::Naive => vec!["--naive".to_owned()],
            Backend::QCIS => vec!["--qcis".to_owned()],
            Backend::QCISGen => vec!["--qcisgen".to_owned()],
        }
    }
}
pub struct ISQSimulator{
    tool: ISQBinTool,
    pub entry: String,
    pub parameters: Parameters,
    pub backend: Backend,
    pub shots: i64,
    pub debug: bool,
    pub probs: bool,
    pub np: i64,
    pub qn: usize
}
impl ISQSimulator{
    pub fn new()->ISQSimulator{
        ISQSimulator{
            tool: ISQBinTool::new("simulator"),
            entry: "__isq__entry".to_owned(),
            parameters: Default::default(),
            backend: Backend::Naive,
            shots: 1,
            debug: false,
            probs: false,
            np: 1,
            qn: 1
        }
    }
    fn args(&self)->Vec<String>{
        let mut args = vec![];
        args.push("-e".to_owned());
        args.push(self.entry.clone());
        for param in self.parameters.param_int.iter(){
            args.push("-i".to_owned());
            args.push(format!("{}", param));
        }
        for param in self.parameters.param_double.iter(){
            args.push("-d".to_owned());
            args.push(format!("{}", param));
        }
        args.extend(self.backend.to_args());
        args.push("--shots".to_owned());
        args.push(self.shots.to_string());
        if self.debug{
            args.push("--debug".to_owned());
        }
        if self.probs{
            args.push("--probs".to_owned());
        }
        args.push("--np".to_owned());
        args.push(self.np.to_string());
        args.push("--qn".to_owned());
        args.push(self.qn.to_string());
        args
    }
    pub fn generate_qcis(&mut self, object: &File, qcis_route_config: Option<&File>)->miette::Result<File>{
        let qcis_file = self.tool.tempfile(qcis())?;
        self.backend = Backend::QCISGen;
        let mut args = self.args();
        args.push(object.path()?.to_owned());
        let mut envs = vec![];
        if let Some(qcis_config) = qcis_route_config{
            envs.push(("QCIS_ROUTE_CONFIG", qcis_config.path()?.to_owned()));
        }
        let qcis_code = self.tool.call_with_env(&args, &envs)?;
        let mut f = qcis_file.open()?;
        write!(f, "{}", qcis_code).map_err(io_error_when("Writing QCIS"))?;
        Ok(qcis_file)

    }
    pub fn run(&self, object: &File)->miette::Result<()>{
        let mut args = self.args();
        args.push(object.path()?.to_owned());
        self.tool.exec_with_env(&args, &[])?;
        Ok(())
    }
}