pub mod isq_simulator;
pub mod isqc1;
pub mod isq_opt;
pub mod llvm;
use std::{ffi::OsStr, io::{Read, Write}, sync::Arc};

use tempfile::NamedTempFile;

use crate::{exec::*, io_error_when, MalformedUTF8Error, NoISQv2RootError};
/**
 * Tool class for invoking a binary tool in $ISQ_ROOT/bin.
 */
pub struct ISQBinTool{
    name: String,
    is_llvm: bool
}

impl ISQBinTool{
    pub fn new(tool: &str)->Self{
        ISQBinTool{
            name: tool.to_string(),
            is_llvm: false
        }
    }
    pub fn new_llvm(tool: &str)->Self{
        ISQBinTool{
            name: tool.to_string(),
            is_llvm: true
        }
    }
    pub fn isq_tool_debug()->bool{
        std::env::var("ISQ_TOOL_DEBUG").is_ok()
    }
    pub fn isq_root()->miette::Result<String>{
        Ok(std::env::var("ISQ_ROOT").map_err(|_| NoISQv2RootError)?)
    }
    pub fn llvm_root()->miette::Result<String>{
        if std::env::var("ISQC_DEV_ENV").is_ok(){
            trace!("DevEnv detected.");
            return Ok(std::env::var("ISQC_DEV_MLIR_ROOT").map_err(|_| NoISQv2RootError)?);
        }
        ISQBinTool::isq_root()
    }
    pub fn root(&self)->miette::Result<String>{
        if self.is_llvm{
            return ISQBinTool::llvm_root();
        }
        ISQBinTool::isq_root()
    }
    pub fn call_with_stdin_and_env<S: AsRef<OsStr>, T: AsRef<str>, U: AsRef<str>>(&self, args: &[S], stdin: &str, env: &[(T, U)])->miette::Result<String>{
        return Ok(exec_command_text_with_decorator(self.root()?.as_str(), &self.name, args, stdin, |child| {
            for (k, v) in env.iter(){
                child.env(k.as_ref(), v.as_ref());
            }
        }).map_err(io_error_when(&format!("Calling \"{}\"", self.name)))?);
    }
    pub fn call_with_stdin<S: AsRef<OsStr>>(&self, args: &[S], stdin: &str)->miette::Result<String>{
        self.call_with_stdin_and_env::<_, &str, &str>(args, stdin, &[])
    }
    pub fn call_with_env<S: AsRef<OsStr>, T: AsRef<str>, U: AsRef<str>>(&self, args: &[S], env: &[(T, U)])->miette::Result<String>{
        return self.call_with_stdin_and_env(args, "", env);
    }
    pub fn call<S: AsRef<OsStr>>(&self, args: &[S])->miette::Result<String>{
        return self.call_with_stdin(args, "");
    }
    /**
     * Call with inherited stdin and stdout
     */
    pub fn exec_with_env<S: AsRef<OsStr>>(&self, args: &[S], envs: &[(String, String)])->miette::Result<()>{
        return Ok(raw_exec_command_decorator(self.root()?.as_str(), &self.name, args, |x|{
            for (k, v) in envs.iter(){
                x.env(k, v);
            }
        }).map_err(io_error_when(&format!("Execing \"{}\"", self.name)))?);
    }
    pub fn tempfile(&self, suffix: &str)->miette::Result<File>{
        Ok(File::TempFile(Arc::new(tempfile::Builder::new().suffix(&format!(".{}", suffix)).tempfile().map_err(io_error_when("Creating tempfile"))?)))
    }
}
//pub type File = tempfile::NamedTempFile;




#[derive(Clone)]
pub enum File{
    ExternalFile(String),
    TempFile(Arc<NamedTempFile>)
}

impl File{
    pub fn path(&self)->miette::Result<&str>{
        match self{
            File::ExternalFile(path)=>Ok(&path),
            File::TempFile(file)=>Ok(file.path().to_str().ok_or_else(|| MalformedUTF8Error)?)
        }
    }
    pub fn open(&self)->miette::Result<std::fs::File>{
        match self{
            File::ExternalFile(path)=>{
                let handle = std::fs::File::open(path).map_err(io_error_when(&format!("Opening file {}", path)))?;
                Ok(handle)
            },
            File::TempFile(file)=>{
                let handle = file.reopen().map_err(io_error_when(&format!("Reopening tempfile {}", file.path().to_str().ok_or_else(|| MalformedUTF8Error)?)))?;
                Ok(handle)
            }
        }
    }
    pub fn copy_to<S: AsRef<OsStr>>(&self, dest: S)->miette::Result<()>{
        let dest = dest.as_ref();
        let src = self.path()?;
        std::fs::copy(src, dest).map_err(io_error_when(&format!("Copying file {} to {}", src, dest.to_str().unwrap())))?;
        Ok(())
    }
}
impl AsRef<File> for File{
    fn as_ref(&self) -> &File {
        self
    }
}
/// Extension names
pub mod ext{
    use super::isq_simulator::ISQSimulatorConfig;
    /// isQ source code
    pub fn isq_src()->&'static str{
        "isq"
    }
    /// MLIR textual form
    pub fn mlir_src()->&'static str{
        "mlir"
    }
    /// MLIR Bytecode (unused)
    pub fn mlir_bc()->&'static str{
        "bc"
    }
    /// LLVM textual form
    pub fn llvm_src()->&'static str{
        "ll"
    }
    /// LLVM bytecode
    pub fn llvm_bc()->&'static str{
        "bc"
    }
    /// Object file
    pub fn object()->&'static str{
        "o"
    }
    /// Binary file, consumed by the simulator.
    pub fn binary()->miette::Result<String>{
        ISQSimulatorConfig::new().dllextension()
    }
    /// OpenQASM2 output
    pub fn qasm()->&'static str{
        "qasm"
    }
    /// OpenQASM3 output
    pub fn qasm3()->&'static str{
        "qasm3"
    }
    /// eQASM output
    pub fn eqasm()->&'static str{
        "eqasm"
    }
    /// QCIS format
    pub fn qcis()->&'static str{
        "qcis"
    }
}