use clap::{Parser, ValueEnum};
use serde::{Deserialize, Serialize};
use std::collections::BTreeMap;
#[derive(Parser)]
#[clap(about, version, author)]
struct Cli {
    /// What mode to run the program in
    #[arg(value_enum)]
    mode: Mode,
}

#[derive(Copy, Clone, PartialEq, Eq, PartialOrd, Ord, ValueEnum)]
enum Mode {
    /// Flags for llvm-link
    LLVMLinkFlags,
    /// Link flags
    LDFlags,
    /// Target triple
    TargetTriple,
    /// Data Layout string
    DataLayout,
    /// DLL suffix
    DLLSuffix,
    /// LLVM IR Header
    LLVMIRHeader,
    /// LLVM stub
    LLVMLinkStub
}
use std::env;
fn default_dll_suffix()->String{
    return ".so".to_string();
}
fn default_dll_prefix()->String{
    return "lib".to_string();
}
#[derive(Serialize, Deserialize)]
struct Target{
    #[serde(rename = "data-layout")]
    dataLayout: String,
    #[serde(rename = "dll-suffix")]
    #[serde(default = "default_dll_suffix")]
    dllSuffix: String,
    #[serde(rename = "dll-prefix")]
    #[serde(default = "default_dll_prefix")]
    dllPrefix: String
}
include!(concat!(env!("OUT_DIR"), "/host_env.rs"));
fn main() {
    let cli = Cli::parse();
    let isq_root = env::var("ISQ_ROOT").unwrap_or("$ISQ_ROOT".to_owned());
    let targetJson : BTreeMap<String, Target> = serde_json::from_str(&std::fs::read_to_string(format!("{}/share/isq-simulator/simulator-targets.json", isq_root)).unwrap()).unwrap();
    let mut host = SIMULATOR_HOST.to_owned();
    let host_info = targetJson.get(&host).unwrap();
    match cli.mode {
        Mode::TargetTriple=>{
            println!("{}", &host);
        }
        Mode::DataLayout => {
            println!("{}", &host_info.dataLayout);
        }
        Mode::DLLSuffix => {
            println!("{}", &host_info.dllSuffix);
        }
        Mode::LLVMLinkFlags => {
            println!("");
        }
        Mode::LLVMLinkStub => {
            println!("{}/share/isq-simulator/isq-simulator.ll", isq_root);
        }
        Mode::LLVMIRHeader => {
            println!(";;;;;;;;;;;;;;;");
            println!("; Target and data layout of isQ simulator shared object.");
            println!("target datalayout = \"{}\"", &host_info.dataLayout);
            println!("target triple = \"{}\"", &host);
            println!(";;;;;;;;;;;;;;;")
        }
        Mode::LDFlags => {
            let mut flags: Vec<String> = vec![
                "-shared".to_owned()
            ];
            #[cfg(feature = "win32")]
            {
                flags.push(format!("-m i386pep"));
                flags.push(format!("{}/lib/isq-simulator/msvcrt.a", isq_root));
                flags.push(format!("{}/lib/isq-simulator/dllcrt.o", isq_root));
                // Win32 DLL symbol resolution requires correct simlator executable name as identifier.
                flags.push(format!("{}/lib/isq-simulator/isq-simulator.a", isq_root));
            }
            println!("{}", flags.join(" "));
        }
    }
}